<?php
/*
	Plugin Name: WM Video Playlists
	Plugin URI: 
	Description: Reduce the bounce rate and keep readers on your site by displaying YouTube and Vimeo videos in awesome playlists
	Author: WatermelonWP.com
	Version: 1.0.2
	Author URI: https://watermelonwp.com/
	License: GPL2+
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


if ( ! class_exists( 'WMVP' ) ) {

	/**
	 * The main WMVP class
	 */
	class WMVP {

		/**
		 * The plugin version number.
		 */
		public $version = '1.0.2';


		/**
		 * A dummy constructor to ensure WMVP is only setup once.
		 */
		public function __construct() {
			// Do nothing.
		}


		/**
		 * Sets up the WMVP plugin.
		 */
		public function initialize() {

			// Define constants.
			$this->define( 'WMVP', true );
			$this->define( 'WMVP_VERSION', $this->version );
			$this->define( 'WMVP_POST_SLUG', 'wmvp-playlist' );
			$this->define( 'WMVP_PLUGIN_FILE', __FILE__ );
			$this->define( 'WMVP_PLUGIN_DIR', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
			$this->define( 'WMVP_PLUGIN_URL', untrailingslashit( plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) ) ) );
			
			// Register activation hook.
			register_activation_hook( __FILE__, array( $this, 'plugin_activated' ) );

			// Include files
			require_once WMVP_PLUGIN_DIR . '/includes/wmvp-helper-functions.php';
			require_once WMVP_PLUGIN_DIR . '/includes/wmvp-functions.php';
			require_once WMVP_PLUGIN_DIR . '/includes/class-wmvp-api.php';

			// Include admin.
			if( is_admin() ){
				require_once WMVP_PLUGIN_DIR . '/includes/admin/class-wmvp-settings.php';
				require_once WMVP_PLUGIN_DIR . '/includes/admin/class-wmvp-admin.php';
			}

			// Include PRO.
			if ( file_exists( WMVP_PLUGIN_DIR . '/pro' ) ) {
				require_once WMVP_PLUGIN_DIR . '/pro/wmvp-pro.php';
			}

			if ( is_admin() && ! wmvp_is_pro() ) {
				require_once WMVP_PLUGIN_DIR . '/includes/admin/wmvp-upgrade-notices.php';
			}

			// Add actions.
			add_action( 'init', array( $this, 'init' ), 5 );
			add_action( 'init', array( $this, 'register_post_types' ), 5 );

			// Load Languages Files
			add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );

			// Register Main Scripts and Styles
			add_action( 'wp_enqueue_scripts', array( $this, 'register_assets' ) );

			add_action( 'wp_head', array( $this, 'head_styles' ) );
		}


		/**
		 * Completes the setup process on "init" of earlier.
		 */
		public function init() {

		}


		/**
		 * Registers the WMVP post types.
		 */
		public function register_post_types() {
		
			register_post_type( WMVP_POST_SLUG,
				array(
					'labels' => array(
						'name'          => esc_html__( 'Video Playlists',  'wm-video-playlists' ),
						'singular_name' => esc_html__( 'Playlist',         'wm-video-playlists' ),
						'add_new_item'  => esc_html__( 'Add New Playlist', 'wm-video-playlists' ),
					),
					'public'              => false,
					'show_ui'             => true,
					'show_in_menu'        => true,
					'menu_icon'           => 'dashicons-playlist-video',
					'menu_position'       => 99,
					'can_export'          => true,
					'exclude_from_search' => true,
					'capability_type'     => 'post',
					'hierarchical'        => false,
					'supports'            => array( 'title' ),
				)
			);
		}


		/**
		 * Defines a constant if doesnt already exist.
		 */
		public function define( $name, $value = true ) {
			if ( ! defined( $name ) ) {
				define( $name, $value );
			}
		}


		/**
		 * Load Languages Files
		 */
		function load_textdomain() {
			load_plugin_textdomain( 'wm-video-playlists', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' ); 
		}


		/**
		 * Register Main Scripts and Styles
		 */
		function register_assets(){

			wp_register_script( 'wmvp-js-videos-list', WMVP_PLUGIN_URL . '/assets/js/videos-playlist.js', array( 'jquery' ), WMVP_VERSION, true );

			wp_enqueue_style( 'wmvp-style-videos-list', WMVP_PLUGIN_URL . '/assets/css/style.css', array(), WMVP_VERSION );
		}


		/**
		 * Plugin Activation Hook
		 */
		public function plugin_activated() {
			
			do_action( 'WMVP/first_activated' );
		
			if ( wmvp_is_pro() ) {
				do_action( 'WMVP/activated_pro' );
			}
		}

				
		/**
		 * Default Light and Dark CSS Vars
		 */
		function css_vars( $theme = false ){
			
			// Light
			if( $theme == 'light' ){
				$vars = array(
					'playlist-head-bg' => '#DCDDFF',
					'playlist-head-color' => '#100024',
					'current-video-bg' => '#f2f4f5',
					'current-video-color' => '#000',
					'playlist-bg' => '#fff',
					'playlist-color' => '#000',
					'playlist-outer-border' => '1px solid rgba(0,0,0,0.07)',
					'playlist-inner-border-color' => 'rgba(0,0,0,0.07)',
				);
			}

			// Dark
			else{
				$vars = array(
					'playlist-head-bg' => '#131416',
					'playlist-head-color' => '#fff',
					'current-video-bg' => '#24262a',
					'current-video-color' => '#fff',
					'playlist-bg' => '#1a1b1f',
					'playlist-color' => '#fff',
					'playlist-outer-border' => '1px solid rgba(0,0,0,0.07)',
					'playlist-inner-border-color' => 'rgba(255,255,255,0.05)',
				);
			}

			return apply_filters( 'WMVP/CSS/vars', $vars, $theme );
		}


		/**
		 * Styles
		 */
		function head_styles( $selector = false ){

			$this->css_skins_vars( 'light' );
			$this->css_skins_vars( 'dark' );
		}


		/**
		 * 
		 */
		function css_skins_vars( $theme ){

			$css_vars = $this->css_vars( $theme );
			$selector = apply_filters( 'WMVP/CSS/selector', ".$theme-skin", $theme ) ;
			$css_out  = '';

			// Custom Colors
			$custom_colors = $this->get_css_vars_keys();

			foreach( $custom_colors as $var => $color ){
				$color = wmvp_get_option( $var, $css_vars[ $var ] );
				$css_out .="--wmvp-$var: $color;";  
			}

			// Print CSS
			if( ! empty( $css_out ) ){
				echo '<style>'. $selector .'{'. $css_out .'}</style>';
			}
		}


		/**
		 * Custom Player Colors
		 */
		function inline_styles( $selector ){

			$css_out = '';

			// Custom Colors
			$custom_colors = $this->get_css_vars_keys();

			foreach( $custom_colors as $var => $color ){
				if( $color = wmvp_get_playlist_meta( 'wmvp-' . $var ) ){
					$css_out .="--wmvp-$var: $color;";  
				}
			}

			// Print CSS
			if( ! empty( $css_out ) ){
				echo '<style>'. $selector .'{'. $css_out .'}</style>';
			}
		}


		/**
		 * Get the list of CSS vars
		 */
		function get_css_vars_keys(){

			return array(
				'playlist-head-bg'            => false,
				'playlist-head-color'         => false,
				'current-video-bg'            => false,
				'current-video-color'         => false,
				'playlist-bg'                 => false,
				'playlist-color'              => false,
				'playlist-outer-border'       => false,
				'playlist-inner-border-color' => false,
			);
		}

	}


	/**
	 * The main function responsible for returning the one true wmvp Instance to functions everywhere.
	 * Use this function like you would a global variable, except without needing to declare the global.
	 *
	 * Example: <?php $wmvp = wmvp(); ?>
	 *
	 * @return  WMVP
	 */
	function wmvp() {
		global $wmvp;

		// Instantiate only once.
		if ( ! isset( $wmvp ) ) {
			$wmvp = new WMVP();
			$wmvp->initialize();
		}
		return $wmvp;
	}

	// Instantiate.
	wmvp();
} // class_exists check



