<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


if( ! class_exists( 'WMVP_Admin' ) ) {

	class WMVP_Admin{


		/**
		 * __construct
		 *
		 * Class constructor where we will call our filter and action hooks.
		 */
		function __construct(){

			// Actions
			add_action( 'admin_init',            array( $this, 'admin_init' ) );

			add_action( 'admin_menu',            array( $this, 'admin_menu' ) );
			add_action( 'add_meta_boxes',        array( $this, 'register_shortcode_metabox' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

			// Playlists List screen
			add_filter( 'display_post_states', array( $this, 'playlist_type_label' ), 10, 2 );

			// Plugins page
			add_filter( 'plugin_action_links', array( $this, 'plugin_action_links' ), 10, 2 );

			// Plylists Add/Edit Screen
			add_action( 'edit_form_after_title',       array( $this, 'playlist_screen' ) );
			add_action( 'WMVP/Playlist_Screen/New',    array( $this, 'add_new_list' ) );
			add_action( 'WMVP/Playlist_Screen/after',  array( $this, 'add_manual_playlist' ), 10, 2 );
			add_action( 'WMVP/Playlist_Screen/after',  array( $this, 'playlist_options' ), 99, 2);
			add_action( 'WMVP/Playlist_Screen/after',  array( $this, 'automatic_playlist_notice' ), 10, 2 );


			add_action( 'save_post', array( $this, 'save_playlist' ) );
		}


		/**
		 * Register the default Settings.
		 */
		function admin_init(){

			// Save the default settings
			if( ! get_option( 'wmvp_version' ) ){
				
				// Default settings
				if( ! get_option( 'wmvp_options' ) ){

					$default_settings = array(
						'style'         => 'dark',
						'layout'        => 'vertical',
						'shadow'        => '1',
						'player_title'  => 'true',
						'date'          => 'true',
						'time_format'   => 'traditional',
						'duration'      => 'true',
						'numbers'       => 'true',
						'thumbnail'     => 'true',
					);

					update_option( 'wmvp_options', $default_settings );
				}
				
				// Store the Plugin Version in DB, Will be used to perform DB updates 
				update_option( 'wmvp_version', WMVP_VERSION, false );
			}
		}


		/**
		 * If current Playlist's type is Automatica but the PRO version is not active
		 */
		function automatic_playlist_notice( $list_type = false, $post_id = false ){

			if( $list_type != 'automatic' || wmvp_is_pro() ){
				return;
			}

			echo '<br />';
			wmvp_notice( sprintf( esc_html__( 'This is an automatic Playlist, Activate the %1sPRO plugin%2s to be able to edit it.', 'wm-video-playlists' ), '<a href="'. wmvp_go_pro() .'" target="_blank"><strong>', '</strong></a>' ) );
		}

		/**
		 * 
		 */
		function add_new_list(){
			?>
		
			<div class="wmvp-section is-wmvp" id="wmvp-list-type-section">
				<div class="wmvp-add-newlist">
		
					<h2><?php esc_html_e( 'Select the Playlist Type', 'wm-video-playlists' ) ?></h2>
					<div class="wmvp-add-videos-type-wrap">
		
						<?php
						$list_types = array(
		
							'manual' => array(
								'title'       => esc_html__( 'Manual', 'wm-video-playlists' ),
								'description' => esc_html__( 'Add videos Manually one by one from YouTube and Vimeo', 'wm-video-playlists' ),
								'image'       => WMVP_PLUGIN_URL .'/includes/admin/assets/images/videos-manual.png',
							),
		
							'youtube-playlist' => array(
								'title'       => esc_html__( 'YouTube Playlist', 'wm-video-playlists' ),
								'description' => esc_html__( 'Import videos automatically from a YouTube Playlist', 'wm-video-playlists' ),
								//'caption'     => 'Display posts from any public hashtag with an Instagram hashtag feed. Great for pulling in user-generated content associated with your brand, running promotional hashtag campaigns, engaging audiences at events, and more.',
								'image'       => WMVP_PLUGIN_URL .'/includes/admin/assets/images/videos-automatic.png',
								'is_pro'      => true,
								'provider'    => 'youtube',
								'type'        => 'playlist',
							),
		
							'youtube-channel' => array(
								'title'       => esc_html__( 'YouTube Channel', 'wm-video-playlists' ),
								'description' => esc_html__( 'Import videos automatically from a YouTube Channel.', 'wm-video-playlists' ),
								'image'       => WMVP_PLUGIN_URL .'/includes/admin/assets/images/videos-automatic.png',
								'is_pro'      => true,
								'provider'    => 'youtube',
								'type'        => 'channel',
							),
		
							'youtube-search' => array(
								'title'       => esc_html__( 'YouTube Search', 'wm-video-playlists' ),
								'description' => esc_html__( 'Import videos automatically from a YouTube search', 'wm-video-playlists' ),
								'image'       => WMVP_PLUGIN_URL .'/includes/admin/assets/images/videos-automatic.png',
								'is_pro'      => true,
								'provider'    => 'youtube',
								'type'        => 'search',
							),
							/*
							'youtube-user' => array(
								'title'       => esc_html__( 'YouTube User', 'wm-video-playlists' ),
								'description' => esc_html__( 'Add videos one by one', 'wm-video-playlists' ),
								'image'       => WMVP_PLUGIN_URL .'/includes/admin/assets/images/videos-automatic.png',
								'is_pro'      => true,
								'provider'    => 'youtube',
								'type'        => 'user',
							),
		
							'youtube-handel' => array(
								'title'       => esc_html__( 'YouTube @handel', 'wm-video-playlists' ),
								'description' => esc_html__( 'Add videos one by one', 'wm-video-playlists' ),
								'image'       => WMVP_PLUGIN_URL .'/includes/admin/assets/images/videos-automatic.png',
								'is_pro'      => true,
								'provider'    => 'youtube',
								'type'        => 'handel',
							),
							
							
							'vimeo-user' => array(
								'title'       => esc_html__( 'Vimeo User', 'wm-video-playlists' ),
								'description' => esc_html__( 'Add videos Manually one by one from Youtube and Vimeo', 'wm-video-playlists' ),
								'image'       => WMVP_PLUGIN_URL .'/includes/admin/assets/images/vimeo-automatic.png',
								'is_pro'      => true,
								'provider'    => 'vimeo',
								'type'        => 'user',
							),
		
							'vimeo-group' => array(
								'title'       => esc_html__( 'Vimeo Group', 'wm-video-playlists' ),
								'description' => esc_html__( 'Add videos one by one', 'wm-video-playlists' ),
								'image'       => WMVP_PLUGIN_URL .'/includes/admin/assets/images/vimeo-automatic.png',
								'is_pro'      => true,
								'provider'    => 'vimeo',
								'type'        => 'group',
							),
		
							'vimeo-channel' => array(
								'title'       => esc_html__( 'Vimeo Channel', 'wm-video-playlists' ),
								'description' => esc_html__( 'Add videos one by one', 'wm-video-playlists' ),
								'image'       => WMVP_PLUGIN_URL .'/includes/admin/assets/images/vimeo-automatic.png',
								'is_pro'      => true,
								'provider'    => 'vimeo',
								'type'        => 'channel',
							),
							*/
						);
		
						$list_types = apply_filters( 'WMVP/Add_New/types', $list_types );
		
						foreach( $list_types as $id => $args ){
		
							$data_atts = array();
							$data_atts[] = "data-upgrade='". esc_attr( sprintf( 'Upgrade to Pro to automatically get videos from a %2s.', $args['title']  ) ) ."'";
							foreach( $args as $data_att => $data_value ){
								$data_atts[] = "data-$data_att='". esc_attr( $data_value ) ."'";
							}
							?>
							<div class="wmvp-add-videos-type" id="wmvp-add-videos-<?php echo esc_attr( $id ) ?>" <?php echo join( ' ', $data_atts ); ?>>
								<div class="wmvp-add-videos-type-img">
									<img src="<?php echo esc_attr( $args['image'] ); ?>" alt="" />
									<?php 
										if( $id != 'manual' ){
											echo '<span class="wmvp-type-label wmvp-type-automatic">'. esc_html__( 'Automatic', 'wm-video-playlists' ) .'</span>'; 
										}
									?>
								</div>
								<div class="wmvp-add-videos-type-info">
									<h3><?php echo esc_html( $args['title'] ); ?></h3>
									<p><?php echo esc_html( $args['description'] ); ?></p>
								</div><!-- .wmvp-add-videos-type-info -->
							</div><!-- .wmvp-add-videos-type -->
							<?php
						}
						?>
		
					</div><!-- .wmvp-add-videos-type-wrap -->
				</div><!-- .wmvp-add-newlist -->
			</div><!-- #wmvp-list-type -->
			<?php 
		}

		
		/**
		 * Add/Edit Playlist Screen
		 */
		function playlist_screen() {

			$post_id      = get_the_id();
			$current_page = get_current_screen();

			if( get_post_type( $post_id ) != WMVP_POST_SLUG || $current_page->post_type != WMVP_POST_SLUG ){
				return;
			}

			$list_type = get_post_meta( $post_id, 'wmvp-list-type', true );
			
			do_action( 'WMVP/Playlist_Screen/before', $list_type, $post_id );

			// Missing API keys
			if( ! get_option( 'wmvp_youtube_key' ) ){
				wmvp_notice( esc_html__( 'You need to set the YouTube API Key in the plugin settings page to be able to add videos.', 'wm-video-playlists' ), 'error' );
			}

			?>	

			<input type="hidden" name="wmvp-nonce" id="wmvp-nonce" value="<?php echo esc_attr( wp_create_nonce( plugin_basename(__FILE__) ) )?>" />
			<input type="hidden" id="wmvp-list-type" name="wmvp-list-type" value="<?php echo esc_attr( $list_type ) ?>" />

			<?php

			// New Playlist
			if( empty( $list_type ) ) {
				do_action( 'WMVP/Playlist_Screen/New', $list_type, $post_id );
			}
			
			do_action( 'WMVP/Playlist_Screen/after', $list_type, $post_id );
		}


		/**
		 * Add Manual Playlist
		 */
		function add_manual_playlist( $list_type = false, $post_id = false ){
			?>
		
			<!-- Manual Settings Section -->
			<div class="wmvp-section is-wmvp" id="wmvp-list-type-manual" <?php if( $list_type == 'manual' ){ echo ' style="display: block"'; } ?>>
				<div class="wmvp-section-head">
					<h3><?php esc_html_e( 'Manual', 'wm-video-playlists' ) ?></h3>
				</div>
				<div class="wmvp-settings-section">
					<div class="wmvp-videos-items-wrapper">
		
						<?php
							$videos_list = get_post_meta( $post_id, 'wmvp_videos_manual', true );
		
							// Add an empty first element if the list is empty
							if( empty( $videos_list ) || ! is_array( $videos_list ) ){
								$videos_list = array(
									'url' => '',
								);
							}
						?>
		
						<ul id="wmvp-videos-items-manual" class="wmvp-videos-items wmvp-videos-items-sortable">
							<?php
		
							$counter = 0;
							foreach( $videos_list as $single_video ){
								$counter++;
								?>
								<li class="wmvp-single-video" id="wmvp-video-item-<?php echo esc_attr( $counter ) ?>">
									<a class="wmvp-icon-move dashicons dashicons-screenoptions"></a>
									<div class="wmvp-video-thumbnail">
										<a href="<?php if( ! empty( $single_video['url'] ) ) { echo esc_url( $single_video['url'] ); } ?>" target="_blank"><img src="<?php echo esc_attr( wmvp_get_video_thumbnail( $single_video ) ); ?>" alt="" width="120" height="90" /></a>
									</div>
									<div class="wmvp-option">
										<label for="wmvp_videos_manual[<?php echo esc_attr( $counter ) ?>][url]">
											<span class="wmvp-label"><?php esc_html_e( 'Video URL:', 'wm-video-playlists' ) ?></span>
											<input class="wmvp-url-field" placeholder="https://" id="wmvp_videos_manual[<?php echo esc_attr( $counter ) ?>][url]" name="wmvp_videos_manual[<?php echo esc_attr( $counter ) ?>][url]" value="<?php if( ! empty( $single_video['url'] ) ) { echo esc_attr( $single_video['url'] ); } ?>" type="text" />
											<p class="wmvp-description"><?php esc_html_e( 'Supports: YouTube and Vimeo', 'wm-video-playlists' ) ?></p>
										</label>
									</div>
									<div class="wmvp-option">
										<label for="wmvp_videos_manual[<?php echo esc_attr( $counter ) ?>][custom_title]">
											<span class="wmvp-label"><?php esc_html_e( 'Custom Title:', 'wm-video-playlists' ) ?>  </span>
											<input id="wmvp_videos_manual[<?php echo esc_attr( $counter ) ?>][custom_title]" name="wmvp_videos_manual[<?php echo esc_attr( $counter ) ?>][custom_title]" value="<?php if( ! empty( $single_video['custom_title'] ) ) { echo esc_attr( $single_video['custom_title'] ); } ?>" placeholder="<?php if( ! empty( $single_video['title'] ) ) { echo esc_attr( $single_video['title'] ); } ?>" type="text" />
											<p class="wmvp-description"><?php esc_html_e( 'Optional', 'wm-video-playlists' ) ?></p>
										</label>
									</div>
									<a class="wmvp-icon-delete dashicons dashicons-minus"></a>
								</li>
								<?php
							}
							?>
						</ul>
		
						<div class="wmvp-section-footer">
							<input id="wmvp-add-video-button" type="button" class="wmvp-button button button-primary" value="<?php esc_html_e( 'Add Video', 'wm-video-playlists' ) ?>">
						</div>
		
					</div> <!-- .wmvp-videos-items-wrapper -->
		
				</div><!-- .wmvp-settings-section -->
			</div><!-- #wmvp-list-type-manual -->
		
			<script>
				var wmvp_Js_Counter = <?php echo esc_js( $counter+1 ) ?>;
		
				jQuery(document).ready(function() {
		
					// Sortable Lists
					jQuery( '.wmvp-videos-items-sortable' ).sortable({ placeholder: 'wmvp-sortable-placeholder' });
		
					// --- 
					function wmvp_Js_YouTubeGetID(url){
						var ID = '';
						url = url.replace(/(>|<)/gi,'').split(/(vi\/|v=|\/v\/|youtu\.be\/|\/embed\/)/);
						if(url[2] !== undefined) {
							ID = url[2].split(/[^0-9a-z_\-]/i);
							ID = ID[0];
							ID = 'https://i.ytimg.com/vi/'+ ID +'/mqdefault.jpg';
						}
						return ID;
					}
		
					// Add a Video
					jQuery('#wmvp-add-video-button').click(function(){
						jQuery( '#wmvp-videos-items-manual' ).append( '\
							<li class="wmvp-single-video" id="wmvp-video-item-'+ wmvp_Js_Counter +'" >\
								<a class="wmvp-icon-move dashicons dashicons-screenoptions"></a>\
								<div class="wmvp-video-thumbnail">\
									<img src="<?php echo esc_attr( WMVP_PLUGIN_URL ) ?>/includes/admin/assets/images/blank.png" alt="" width="120" height="90" />\
								</div>\
								<div class="wmvp-option">\
									<label for="wmvp_videos_manual['+ wmvp_Js_Counter +'][url]">\
										<span class="wmvp-label"><?php esc_html_e( 'Video URL:', 'wm-video-playlists' ) ?></span>\
										<input class="wmvp-url-field" placeholder="https://" id="wmvp_videos_manual['+ wmvp_Js_Counter +'][url]" name="wmvp_videos_manual['+ wmvp_Js_Counter +'][url]" value="" type="text" />\
										<p class="wmvp-description"><?php esc_html_e( 'Supports: YouTube and Vimeo', 'wm-video-playlists' ) ?></p>\
									</label>\
								</div>\
								<div class="wmvp-option">\
									<label for="wmvp_videos_manual['+ wmvp_Js_Counter +'][custom_title]">\
										<span class="wmvp-label"><?php esc_html_e( 'Custom Title:', 'wm-video-playlists' ) ?></span>\
										<input id="wmvp_videos_manual['+ wmvp_Js_Counter +'][custom_title]" name="wmvp_videos_manual['+ wmvp_Js_Counter +'][custom_title]" value="" type="text" />\
										<p class="wmvp-description"><?php esc_html_e( 'Optional', 'wm-video-playlists' ) ?></p>\
									</label>\
								</div>\
								<a class="wmvp-icon-delete dashicons dashicons-minus"></a>\
							</li>\
						');
						wmvp_Js_Counter++;		
					});
		
					jQuery(document).on('change', '.wmvp-url-field', function(){
						var $thisElement = jQuery(this),
								theImage     = wmvp_Js_YouTubeGetID( $thisElement.val() );
		
						theImage = theImage ? theImage : '<?php echo esc_attr( WMVP_PLUGIN_URL ) ?>/includes/admin/assets/images/blank.png';
						$thisElement.closest('.wmvp-single-video').find('.wmvp-video-thumbnail img').attr( 'src', theImage );
					});
				});
			</script>
			<?php
		}


		/**
		 * Customizations
		 */
		function playlist_options( $list_type = false, $post_id = false ){

			$settings = $this->get_settings();

			if( empty( $settings ) ){
				return;
			}
			
			?>

			<div id="wmvp-list-type-customizations" <?php if( ! empty( $list_type ) ){ echo ' style="display: block"'; } ?>>

				<h3 style="margin-top: 50px;color: #a0a0a0;font-weight: 400;text-shadow: 1px 1px #fff;"><?php esc_html_e( 'Customizations.', 'wm-video-playlists' ); ?></h3>

				<?php foreach ( $settings as $id => $settings_section ) {

					if( $id != 'player' && $id != 'styles' ) {
						continue;
					}

					if( ! wmvp_is_pro() ){
						$settings_section['is_pro'] = true;
					}

					$settings_section = apply_filters( 'WMVP/Settings/'. $id , $settings_section );
					
					do_action( 'WMVP/Settings/Section/before', $id ); ?>

					<div id="wmvp-id-<?php echo esc_html( $id ); ?>" class="wmvp-section is-wmvp <?php if( ! empty( $settings_section['is_pro'] ) && ! wmvp_is_pro() ) { echo 'wmpv-pro-section'; }  ?>">
		
						<div class="wmvp-section-head"><h3><?php echo $settings_section['name']; if( ! empty( $settings_section['is_pro'] ) && ! wmvp_is_pro() ) { wmvp_pro_label(); } ?></h3></div>

						<div class="wmvp-inside">
							<table class="form-table parent-settings">
								<?php
									foreach ( $settings_section['settings'] as $option ) {

										// Change the settings type to add 'Default' option to inhirit from the global settings
										if( ! empty( $option['type'] ) ){

											$option['id'] = 'wmvp-' . $option['id'];

											// The all section is PRO so no need to add it in the options
											$option['is_pro'] = false;

											switch ( $option['type'] ) {
												case 'radio':
												case 'select':
													if( ! empty( $option['options'] ) ){
														$option['options'] = array_merge( array( '' => esc_html__( 'Default', 'wm-video-playlists' ) ), $option['options'] );
													}
													break;

												case 'checkbox':
													$option['type'] = 'select';
													$option['options'] = array(
														''    => esc_html__( 'Default', 'wm-video-playlists' ),
														'yes' => esc_html__( 'Yes', 'wm-video-playlists' ),
														'no'  => esc_html__( 'No', 'wm-video-playlists' ),
													);														
													break;
											}
										}

										$id = ! empty( $option['id'] ) ? $option['id'] : '';
										wmvp_build_option( $option, $id, wmvp_get_playlist_meta( $id ) );
									}
								?>
							</table>
						</div><!-- .wmvp-inside -->
						<?php
							if( ! empty( $settings_section['is_pro'] ) && ! wmvp_is_pro() ){
								echo '<div class="wmpv-pro-mask"></div>';
							} 
						?>
					</div><!-- .wmvp-section -->
					<?php

					do_action( 'WMVP/Settings/Section/after', $id  );
				}

			?>
			</div><!-- #wmvp-list-type-customizations -->

			<?php
		}


		/**
		 * Save the playlist
		 */
		function save_playlist( $post_id ){

			if( empty( $post_id ) ) {
				return false;
			}

			// Post is saving somehow with our meta box not shown.
			if ( ! isset( $_POST['wmvp-nonce'] ) ) {
				return $post_id;
			}

			// Verify the nonce.
			if ( ! wp_verify_nonce( sanitize_key( $_POST['wmvp-nonce'] ), plugin_basename( __FILE__ ) ) ) {
				return $post_id;
			}

			// Don't try to update meta fields on AUTOSAVE.
			if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
				return $post_id;
			}

			// Manual List
			if( ! empty( $_POST['wmvp_videos_manual'] ) && is_array( $_POST['wmvp_videos_manual'] ) ) {

				$videos = array_map( 'array_filter', wp_unslash( $_POST['wmvp_videos_manual'] ) );
					
				update_post_meta( $post_id, 'wmvp_videos_manual', apply_filters( 'WMVP/Videos/Post/List', $videos ) );
			}
			else{
				delete_post_meta( $post_id, 'wmvp_videos_manual' );
			}

			// Save custom options
			$meta_fields = apply_filters( 'WMVP/post_options_meta', array(
				'wmvp-list-type',
			));

			foreach( $meta_fields as $field ){
				if( ! empty( $_POST[ $field ] ) ) {
					update_post_meta( $post_id, $field, sanitize_text_field( wp_unslash( $_POST[ $field ] ) ) );
				}
				else{
					delete_post_meta( $post_id, $field );
				}
			}

			// ---
			do_action( 'WMVP/after_save', $post_id );
		}



		/**
		 * Add playlist type in the Playlists list page 
		 */
		function playlist_type_label( $post_states, $post ){

			if( isset( $post->post_type ) && $post->post_type == WMVP_POST_SLUG ){

				$type  = get_post_meta( $post->ID, 'wmvp-list-type', true );
				$label = ( $type == 'automatic' ) ? esc_html__( 'Automatic', 'wm-video-playlists' ) : esc_html__( 'Manual', 'wm-video-playlists' );
				$post_states['wmvp_type'] = '<span class="wmvp-type-label wmvp-type-'. $type .'">'. $label .'</span>';
			}

			return $post_states;
		}


		/**
		 * enqueue_scripts
		 *
		 * Register main Scripts and Styles
		 */
		function enqueue_scripts(){

			$screen = get_current_screen();
			if( ! empty( $screen ) && $screen->post_type == WMVP_POST_SLUG ){

				// Styles
				wp_enqueue_style( 'wm-videos-admin-styles', WMVP_PLUGIN_URL . '/includes/admin/assets/css/style.css', array(), time(), 'all' );

				// Scripts
				wp_enqueue_script( 'wm-videos-admin-scripts', WMVP_PLUGIN_URL . '/includes/admin/assets/js/scripts.js', array( 'jquery' ), time(), false );
			
				$js_vars = array(
					'ajaxurl'  => esc_url( admin_url('admin-ajax.php') ),
					'security' => wp_create_nonce( WMVP_PLUGIN_DIR ),
					'is_pro'   => wmvp_is_pro(),
				);

				wp_localize_script( 'wm-videos-admin-scripts', 'wmvp_js', apply_filters( 'WMVP/localize_script', $js_vars ) );
			}
		}


		/**
		 * admin_menu
		 */
		function admin_menu(){

			add_submenu_page(
				'edit.php?post_type='. WMVP_POST_SLUG, 
				esc_html__( 'Settings', 'wm-video-playlists' ),
				esc_html__( 'Settings', 'wm-video-playlists' ),
				'manage_options',
				WMVP_POST_SLUG .'-settings',
				array( $this, 'plugin_options' )
			);

			if( ! wmvp_is_pro() ){
				add_submenu_page(
					'edit.php?post_type='. WMVP_POST_SLUG, 
					esc_html__( 'Upgrade to PRO', 'wm-video-playlists' ),
					'<span style="color: #07EA95;">🍉 '. esc_html__( 'Upgrade to PRO', 'wm-video-playlists' ) .'</span>',
					'manage_options',
					WMVP_POST_SLUG .'-go-pro',
					array( $this, 'upgrade_to_pro' )
				);
			}
		}


		/**
		 * upgrade_to_pro
		 */
		function upgrade_to_pro(){
			echo "<script type='text/javascript'>window.location='". wmvp_go_pro( array( 'utm_medium' => 'menu-link' ) ) ."';</script>";
		}


		/**
		 * Plugins row action links
		 */
		function plugin_action_links( $links, $file ) {
			
			// Trigger The standard and PRO version
			if( strpos( $file, 'wm-video-playlists' ) === false ) {
				return $links;
			}

			$extra_links = array();

			// Upgrade to PRO link
			// '/' to trigger the standard version only
			if( ! wmvp_is_pro() && strpos( $file, 'wm-video-playlists/' ) !== false ) {

				$url = wmvp_go_pro(
					array(
						'utm_medium'  => 'all-plugins',
						'utm_content' => 'upgrade-to-pro',
					)
				);

				$extra_links['wmvp-pro-upgrade'] = sprintf( '<a href="%s" style="color:#00C078;font-weight: 600;" target="_blank">' . esc_html__( 'Upgrade to PRO', 'wm-video-playlists' ) . '</a>', $url );
			}
		
			// Add the Settings page
			$extra_links['wmvp-settings'] = '<a href="' . esc_url( wpvp_get_admin_url() ) . '">' . esc_html__( 'Settings', 'wm-video-playlists' ) . '</a>';

			// Return array of links.
			return array_merge( $extra_links, $links );
		}


		/**
		 * Register the Shortcode Metabox
		 */
		function register_shortcode_metabox(){
			
			$the_post = get_post( get_the_id() );

			if( empty( $the_post->post_status ) || $the_post->post_status != 'publish' ){
				return;
			}

			add_meta_box(
				'wmvp_shortcode_metabox',
				esc_html__( 'Shortcode', 'wm-video-playlists' ),
				array( $this, 'shortcode_metabox' ),
				WMVP_POST_SLUG,
				'side',
				'low'
			);
		}


		/**
		 * The Shortcode Metabox content
		 */
		function shortcode_metabox(){
			echo "
			<input style='
				font-family: Inconsolata, Monaco,MonacoRegular,monospace;
				font-size: 12px;
				background-color:rgb(246, 246, 246);
				border: 1px solid #d9d9d9;
				margin-top: 10px;
				padding: 10px;
				color: #000;
				border-radius: 5px;
				width: 100%' disabled='disabled' value='[wmvp_playlist id=". esc_html( get_the_id() ) ."]' />";
		}

		
		/**
		 * Build Plugin option
		 */
		function build_option( $option ){

			if( empty( $option['id'] )){
				$option['id'] = ' ';
			}

			$value = ! empty( $option['value'] ) ? $option['value'] : wmvp_get_option( $option['id'] ) ;

			wmvp_build_option( $option, 'wmvp_options['.$option["id"].']', $value );
		}


		/**
		 * get_settings
		 *
		 * settings array
		 */
		function get_settings(){


			$settings = array(

				// Vendors APIs
				'api_keys' => array(
					'name'     => esc_html__( 'API', 'wm-video-playlists' ),
					'settings' => array(
						'youtube' => array(
							'name'  => esc_html__( 'YouTube API Key', 'wm-video-playlists' ),
							'id'    => 'youtube_api_key',
							'type'  => 'text',
							'after' => '<button id="wmvp-youtube-api-key" class="wmvp-button button button-primary">'. esc_html__( 'Verify', 'wm-video-playlists' ) .'</button><div class="spinner wmvp-loader"></div><div class="wmvp-notice wmvp-notice-minimal wmvp-notice-error" style="display: none"></div>',
							'value' => get_option( 'wmvp_youtube_key' ),
						),
					)
				),

				// Player
				'player' => array(
					'name'     => esc_html__( 'Player', 'wm-video-playlists' ),
					'settings' => array(
	
						array(
							'name'    => esc_html__( 'Style', 'wm-video-playlists' ),
							'id'      => 'style',
							'type'    => 'radio',
							'options' => array(
								''      => esc_html__( 'Default' ,'wm-video-playlists'),
								'light' => esc_html__( 'Light' ,'wm-video-playlists'),
								'dark'  => esc_html__( 'Dark' ,'wm-video-playlists'),
						)),
						array(
							'name'    => esc_html__( 'Layout', 'wm-video-playlists' ),
							'id'      => 'layout',
							'type'    => 'radio',
							'is_pro'  => true,
							'options' => array(
								'vertical'       => esc_html__( 'Vertical' ,'wm-video-playlists'),
								'horizontal'     => esc_html__( 'Horizontal' ,'wm-video-playlists'),
								'horizontal-nav' => esc_html__( 'Horizontal with floating Nav' ,'wm-video-playlists'),
						)),
	
						array(
							'name'    => esc_html__( 'Shadow', 'wm-video-playlists' ),
							'id'      => 'shadow',
							'type'    => 'select',
							'is_pro'  => true,
							'options' => array(
								'no'    => esc_html__( 'Disable' ,'wm-video-playlists'),
								'smart' => esc_html__( 'Smart' ,'wm-video-playlists'),
								/* translators: $s will be replaced by the shadow #number. */
								'1' => sprintf( esc_html__( 'Shadow %s', 'wm-video-playlists' ), 1 ),
								/* translators: $s will be replaced by the shadow #number. */
								'2' => sprintf( esc_html__( 'Shadow %s', 'wm-video-playlists' ), 2 ),
								/* translators: $s will be replaced by the shadow #number. */
								'3' => sprintf( esc_html__( 'Shadow %s', 'wm-video-playlists' ), 3 ),
								/* translators: $s will be replaced by the shadow #number. */
								'4' => sprintf( esc_html__( 'Shadow %s', 'wm-video-playlists' ), 4 ),
								/* translators: $s will be replaced by the shadow #number. */
								'5' => sprintf( esc_html__( 'Shadow %s', 'wm-video-playlists' ), 5 ),
								/* translators: $s will be replaced by the shadow #number. */
								'6' => sprintf( esc_html__( 'Shadow %s', 'wm-video-playlists' ), 6 ),
						)),
	
						array(
							'name' => esc_html__( 'Enable Player Title', 'wm-video-playlists' ),
							'id'   => 'player_title',
							'type' => 'checkbox',
						),
	
						array(
							'name' => esc_html__( 'Enable Publishing Date', 'wm-video-playlists' ),
							'id'   => 'date',
							'type' => 'checkbox',
						),
	
						array(
							'name'    => esc_html__( 'Date format', 'wm-video-playlists' ),
							'id'      => 'time_format',
							'type'    => 'radio',
							'options' => array(
								'traditional' => esc_html__( 'Traditional' ,'wm-video-playlists'),
								'modern'      => esc_html__( 'Time Ago Format' ,'wm-video-playlists'),
						)),
	
						array(
							'name' => esc_html__( 'Enable Video Duration', 'wm-video-playlists' ),
							'id'   => 'duration',
							'type' => 'checkbox',
						),
						
						array(
							'name' => esc_html__( 'Enable Numbers', 'wm-video-playlists' ),
							'id'   => 'numbers',
							'type' => 'checkbox',
						),
	
						array(
							'name'   => esc_html__( 'Enable Thumbnails', 'wm-video-playlists' ),
							'id'     => 'thumbnail',
							'type'   => 'checkbox',
							'is_pro' => true,
						),
	
						array(
							'name'    => esc_html__( 'Thumbnail Style', 'wm-video-playlists' ),
							'id'      => 'thumbnail_style',
							'type'    => 'radio',
							'is_pro'  => true,
							'options' => array(
								'square' => esc_html__( 'Square' ,'wm-video-playlists'),
								'round'  => esc_html__( 'Rounded' ,'wm-video-playlists'),
								'circle' => esc_html__( 'Circled' ,'wm-video-playlists'),
						)),
	
						array(
							'name'    => esc_html__( 'Limit Videos Titles by', 'wm-video-playlists' ),
							'id'      => 'trim_type',
							'type'    => 'radio',
							'is_pro'  => true,
							'options' => array(
								'words' => esc_html__( 'Words' ,'wm-video-playlists'),
								'chars' => esc_html__( 'Characters' ,'wm-video-playlists'),
						)),
	
						array(
							'name'   => esc_html__( 'Videos Titles Limit', 'wm-video-playlists' ),
							'id'     => 'trim_limit',
							'type'   => 'number',
							'is_pro' => true,
						),
	
					)
				),

				// Styles
				'styles' => array(
					'name'   => esc_html__( 'Styles', 'wm-video-playlists' ),
					'is_pro' => true,
					'settings' => array(
						array(
							'name' => sprintf( esc_html__( 'Playlist Head Section: %s', 'wm-video-playlists' ), '<em>'. esc_html__( 'Background Color', 'wm-video-playlists' ) .'</em>' ),
							'id'   => 'playlist-head-bg',
							'type' => 'color',
						),
	
						array(
							'name' => sprintf( esc_html__( 'Playlist Head Section: %s', 'wm-video-playlists' ), '<em>'. esc_html__( 'Text Color', 'wm-video-playlists' ) .'</em>' ),
							'id'   => 'playlist-head-color',
							'type' => 'color',
						),
	
						array(
							'name' => sprintf( esc_html__( 'Playlist Column: %s', 'wm-video-playlists' ), '<em>'. esc_html__( 'Background Color', 'wm-video-playlists' ) .'</em>' ),
							'id'   => 'playlist-bg',
							'type' => 'color',
						),
	
						array(
							'name' => sprintf( esc_html__( 'Playlist Column: %s', 'wm-video-playlists' ), '<em>'. esc_html__( 'Text Color', 'wm-video-playlists' ) .'</em>' ),
							'id'   => 'playlist-color',
							'type' => 'color',
						),
	
						array(
							'name' => sprintf( esc_html__( 'Playlist Column: %s', 'wm-video-playlists' ), '<em>'. esc_html__( 'Border Color', 'wm-video-playlists' ) .'</em>' ),
							'id'   => 'playlist-inner-border-color',
							'type' => 'color',
						),
	
						array(
							'name' => sprintf( esc_html__( 'Current Playing Video: %s', 'wm-video-playlists' ), '<em>'. esc_html__( 'Background Color', 'wm-video-playlists' ) .'</em>' ),
							'id'   => 'current-video-bg',
							'type' => 'color',
						),
	
						array(
							'name' => sprintf( esc_html__( 'Current Playing Video: %s', 'wm-video-playlists' ), '<em>'. esc_html__( 'Text Color', 'wm-video-playlists' ) .'</em>' ),
							'id'   => 'current-video-color',
							'type' => 'color',
						),
					)
				)
			);

			// Alter plugin settings
			return apply_filters( 'WMVP/Settings', $settings );
		}


		/**
		 * plugin_options
		 *
		 * Plugin Options page
		 */
		function plugin_options(){
	
			// Saving
			if( isset( $_REQUEST['action'] ) && 'wmvp-save' == $_REQUEST['action'] && isset( $_REQUEST['wmvp_options'] ) && isset( $_REQUEST['wmvp-save-nonce'] ) && wp_verify_nonce( sanitize_key( $_REQUEST['wmvp-save-nonce'] ), plugin_basename( __FILE__ ) ) ){
				
				$options = array_map( 'sanitize_text_field', wp_unslash( $_REQUEST['wmvp_options'] ) );

				if( isset( $options ) ){
					unset( $options['license_key'] );
					unset( $options['youtube_api_key'] );
				}
				
				update_option( 'wmvp_options', $options );

				$options_saved = true;
				$saved_message = esc_html__( 'Your changes have been saved.', 'wm-video-playlists' );
			}
			
			// Deactivate License
			elseif( isset( $_REQUEST['deactivate-license'] ) && check_admin_referer( 'deactivate-license', 'deactivate_license_nounce' ) ){
				delete_option( 'wmvp_license_key' );
				delete_option( 'wmvp_license_expire' );

				$options_saved = true;
				$saved_message = esc_html__( 'Your license has been deactivated.', 'wm-video-playlists' );
			}

			$plugin_settings = $this->get_settings();
		?>

		<form method="post">
			<div class="wrap wmvp-settings-page is-wmvp">

				<div class="wmvp-settings-header-wrap">
					<div class="wmvp-settings-header">

						<div class="wmvp-settings-header-row">
							<div class="wmvp-settings-header-title">
								<?php do_action('WMVP/Settings/Header/before_title'); ?>
								<h1><?php echo apply_filters( 'WMVP/plugin_name', esc_html__( 'WM Video Playlists', 'wm-video-playlists' ) ) ?></h1>
								<?php do_action('WMVP/Settings/Header/after_title'); ?>
							</div>
							<div class="wmvp-settings-header-action">
								<input type="hidden" name="action" value="wmvp-save" />
								<input type="hidden" name="wmvp-save-nonce" value="<?php echo esc_attr( wp_create_nonce( plugin_basename( __FILE__ ) ) ) ?>" />
								<input name="save" class="wmvp-button button button-primary" type="submit" value="<?php esc_html_e( 'Save Settings', 'wm-video-playlists' ) ?>" />
							</div>
						</div><!-- .wmvp-settings-header-row -->

					</div> <!-- .wmvp-settings-header -->
				</div> <!-- .wmvp-settings-header-wrap -->

				<?php
					if ( isset( $options_saved ) ) {
						wmvp_notice( $saved_message, 'success' );
					}
				?>

				<div class="wmvp-settings-page-content-wrap">

					<div class="wmvp-settings-page-content">
						<?php

						do_action( 'WMVP/Settings/before' );

						foreach ( $plugin_settings as $id => $settings_section ) {
							
							$settings_section = apply_filters( 'WMVP/Settings/'. $id , $settings_section );
							
							do_action( 'WMVP/Settings/Section/before', $id  ); ?>

							<div id="wmvp-id-<?php echo esc_html( $id ); ?>" class="wmvp-section is-wmvp <?php if( ! empty( $settings_section['is_pro'] ) && ! wmvp_is_pro() ) { echo 'wmpv-pro-section'; }  ?>">
				
								<div class="wmvp-section-head"><h3><?php echo $settings_section['name']; if( ! empty( $settings_section['is_pro'] ) && ! wmvp_is_pro() ) { wmvp_pro_label(); } ?></h3></div>

								<div class="wmvp-inside">
									<table class="form-table parent-settings">
										<?php
											foreach ( $settings_section['settings'] as $option ) {
												$this->build_option( $option );
											}
										?>
									</table>
								</div><!-- .wmvp-inside -->
								<?php
									if( ! empty( $settings_section['is_pro'] ) && ! wmvp_is_pro() ){
										echo '<div class="wmpv-pro-mask"></div>';
									} 
								?>
							</div><!-- .wmvp-section -->

							<?php

							do_action( 'WMVP/Settings/Section/after', $id  );
						}

						do_action( 'WMVP/Settings/after' );
						?>
					</div><!-- .wmvp-settings-page-content -->

					<div id="wmvp-settings-sidebar">
						<?php do_action( 'WMVP/Settings/Sidebar' ); ?>
					</div><!-- #wmvp-settings-sidebar -->

				</div><!-- .wmvp-settings-page-content-wrap -->
			</div><!-- .wmvp-settings-page -->
		</form>

		<?php

		}

	}



	/**
	 * The main function responsible for returning the one true wmvp_admin Instance to functions everywhere.
	 * Use this function like you would a global variable, except without needing to declare the global.
	 *
	 * Example: <?php $wmvp_admin = wmvp_admin(); ?>
	 *
	 * @return  WMVP_Admin
	 */
	function wmvp_admin() {
		global $wmvp_admin;

		// Instantiate only once.
		if ( ! isset( $wmvp_admin ) ) {
			$wmvp_admin = new WMVP_Admin();
		}
		return $wmvp_admin;
	}

	// Instantiate.
	wmvp_admin();
}

