<?php
require_once __DIR__ . '/../config.php';

// ✅ Ensure BASE_API_URL exists
if (!defined('BASE_API_URL')) {
    define('BASE_API_URL', rtrim(API_URL, '/'));
}

// Pagination setup
$page  = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$limit = 24;

// ✅ Get letter slug from rewrite rule
$letter = $_GET['id'] ?? ''; // e.g. A, B, C...
$cleanLetter = strtoupper(preg_replace('/[^A-Z]/i', '', $letter));

// ✅ Validate letter
if (empty($cleanLetter)) {
    die("Invalid or missing letter parameter.");
}

// ✅ Build API endpoint correctly
$endpoint = BASE_API_URL . "/atoz/?letter=" . urlencode($cleanLetter);

// ✅ Function to fetch API data safely
function fetch_api_page($url)
{
    $response = @file_get_contents($url);
    if ($response === false) return null;

    $data = json_decode($response, true);
    if (!empty($data['status']) && $data['status'] === 'success' && !empty($data['data'])) {
        return $data;
    }
    return null;
}

// Initialize variables
$errorMessage  = '';
$searchResults = [];
$currentPage   = $page;
$totalPages    = 1;
$totalCount    = 0;
$seenTmdbIds   = [];
$maxFetchPages = 5; // Prevent overfetch loops

try {
    $pageToFetch = $page;

    // ✅ Keep fetching until 24 unique results or end of pages
    while (count($searchResults) < $limit && $pageToFetch < $page + $maxFetchPages) {
        $apiUrl = "{$endpoint}&page={$pageToFetch}&limit={$limit}";
        $data = fetch_api_page($apiUrl);

        if (!$data) {
            if ($pageToFetch === $page) {
                $errorMessage = "No data found or API error. URL: " . htmlspecialchars($apiUrl);
            }
            break;
        }

        $totalPages  = $data['total_pages'] ?? $totalPages;
        $totalCount  = $data['count'] ?? $totalCount;
        $currentPage = $data['current_page'] ?? $pageToFetch;

        // ✅ Deduplicate by TMDB ID
        foreach ($data['data'] as $item) {
            $tmdbId = $item['tmdb']['tmdb_id'] ?? null;
            if ($tmdbId && !isset($seenTmdbIds[$tmdbId])) {
                $seenTmdbIds[$tmdbId] = true;
                $searchResults[] = $item;
                if (count($searchResults) >= $limit) break 2;
            }
        }

        if ($pageToFetch >= $totalPages) break;
        $pageToFetch++;
    }

    if (empty($searchResults)) {
        $errorMessage = 'No results found for this letter.';
    }
} catch (Exception $e) {
    $errorMessage = 'Error: ' . $e->getMessage();
}

// ✅ Image constants
if (!defined("TMDB_POSTER_BASE")) define("TMDB_POSTER_BASE", "https://image.tmdb.org/t/p/w500");

// ✅ Optional random sections
function fetchRandom($url)
{
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10,
        CURLOPT_FOLLOWLOCATION => true,
    ]);
    $response = curl_exec($ch);
    curl_close($ch);

    $data = json_decode($response, true);
    return $data['data'] ?? [];
}

$randomSeriesList = fetchRandom(BASE_API_URL . "/random?limit=6&type=series");
$randomMovieList  = fetchRandom(BASE_API_URL . "/random?limit=6&type=movie");
?>

<!doctype html>
<html lang="en-US" prefix="og: https://ogp.me/ns#">
<head>
<title><?=$cleanLetter ?> - Toonstream</title>

<meta name="twitter:title" content="<?=$cleanLetter ?> - Toonstream" />
<meta property="og:title" content="<?=$cleanLetter ?> - Toonstream" />
<meta name="twitter:label1" content="<?=$cleanLetter ?>" />
    
    <?php require_once ROOT_PATH . '/components/meta.php'; ?>
</head>
<body class="archive post-type-archive post-type-archive-movies wp-custom-logo wp-theme-torofilm side-right login-off">
<div id="aa-wp" class="cont">
    <?php require_once ROOT_PATH . '/components/header.php'; ?>

    <div class="bd">
        <div class="dfxc">
            <main class="main-site">
                <section class="section movies">
                    <?php require_once ROOT_PATH . '/components/a2z-header.php'; ?>

                    <div class="aa-cn" id="aa-movies">
                        <div id="movies-a" class="aa-tb hdd on">
                            <ul class="post-lst rw sm rcl2 rcl3a rcl4b rcl3c rcl4d rcl6e">
                                <?php if (!empty($searchResults)): ?>
                                    <?php foreach ($searchResults as $item): ?>
                                        <?php
                                        $tmdb   = $item['tmdb'] ?? [];
                                        $title  = htmlspecialchars($tmdb['title'] ?? 'Unknown');
                                        $poster = !empty($tmdb['poster'])
                                            ? TMDB_POSTER_BASE . $tmdb['poster']
                                            : '/assets/no-image.jpg';
                                        $rating = isset($tmdb['rating']) ? round($tmdb['rating'], 1) : 'N/A';
                                        $type   = $tmdb['type'] ?? 'unknown';
                                        $typeSlug = ($type === 'movie') ? 'movies' : $type;
                                        $url = '/' . htmlspecialchars($typeSlug) . '/' . htmlspecialchars($tmdb['url'] ?? '#');
                                        ?>
                                        <li class="post <?= htmlspecialchars($type) ?>">
                                            <article class="post dfx fcl <?= htmlspecialchars($type) ?>">
                                                <header class="entry-header">
                                                    <h2 class="entry-title"><?= $title ?></h2>
                                                    <div class="entry-meta">
                                                        <span class="vote"><span>TMDB</span> <?= $rating ?></span>
                                                    </div>
                                                </header>
                                                <div class="post-thumbnail or-1">
                                                    <figure>
                                                        <img loading="lazy" src="<?= $poster ?>" alt="<?= $title ?>">
                                                    </figure>
                                                    <span class="watch btn sm">View <?= ucfirst($type) ?></span>
                                                    <span class="play fa-play"></span>
                                                </div>
                                                <a href="<?= $url ?>" class="lnk-blk"></a>
                                            </article>
                                        </li>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <p style="text-align:center;">No results found.</p>
                                <?php endif; ?>
                            </ul>
                        </div>
                    </div>

                    <?php if ($totalPages > 1): ?>
                        <?php
                        // ✅ Proper pagination for A–Z pages
                        $pageUrl = function ($p) use ($cleanLetter) {
                            return "/letter/" . urlencode($cleanLetter) . "/page/" . (int)$p;
                        };

                        $maxLinks = 3;
                        $radius = intdiv($maxLinks - 1, 2);
                        $start = max(1, $currentPage - $radius);
                        $end = min($totalPages, $start + $maxLinks - 1);
                        $start = max(1, min($start, $totalPages - $maxLinks + 1));
                        ?>
                        <nav class="navigation pagination">
                            <div class="nav-links">
                                <?php if ($currentPage > 1): ?>
                                    <a href="<?= $pageUrl($currentPage - 1) ?>">PREV</a>
                                <?php endif; ?>

                                <?php if ($start > 1): ?>
                                    <a class="page-link" href="<?= $pageUrl(1) ?>">1</a>
                                    <?php if ($start > 2): ?><a class="extend">...</a><?php endif; ?>
                                <?php endif; ?>

                                <?php for ($i = $start; $i <= $end; $i++): ?>
                                    <a class="page-link <?= $i === $currentPage ? 'current' : '' ?>"
                                       href="<?= $pageUrl($i) ?>"><?= $i ?></a>
                                <?php endfor; ?>

                                <?php if ($end < $totalPages): ?>
                                    <?php if ($end < $totalPages - 1): ?><a class="extend">...</a><?php endif; ?>
                                    <a class="page-link" href="<?= $pageUrl($totalPages) ?>"><?= $totalPages ?></a>
                                <?php endif; ?>

                                <?php if ($currentPage < $totalPages): ?>
                                    <a href="<?= $pageUrl($currentPage + 1) ?>">NEXT</a>
                                <?php endif; ?>
                            </div>
                        </nav>
                    <?php endif; ?>
                </section>
            </main>

            <aside class="sidebar">
                <?php require_once ROOT_PATH . '/components/sidebar.php'; ?>
            </aside>
        </div>

        <?php require_once ROOT_PATH . '/components/footer.php'; ?>
        <div class="bgft"><img loading="lazy" src="/public/img/poster.jpg" alt=""></div>
    </div>
</div>
</body>
</html>
