<?php
require_once __DIR__ . '/../config.php';

if (empty($_GET['id'])) {
    die(json_encode(["status" => "error", "message" => "Missing ID parameter"]));
}

$id_raw = trim($_GET['id']);
$season = null;
$episode = isset($_GET['ep']) ? (int)$_GET['ep'] : null;

// Detect -1x3 pattern for series
if (preg_match('/-(\d+)x(\d+)$/i', $id_raw, $matches)) {
    $season = (int)$matches[1];
    $episode = (int)$matches[2];
}
$id = preg_replace('/-\d+x\d+$/i', '', $id_raw);

define('BASE_API_URL', rtrim(API_URL, '/'));

// Fetch details
$detailsUrl = BASE_API_URL . '/details/?id=' . urlencode($id);
$detailsData = json_decode(@file_get_contents($detailsUrl), true);

if (!$detailsData || $detailsData['status'] !== 'success') {
    die(json_encode(["status" => "error", "message" => "Failed to fetch details data"]));
}

$tmdb = $detailsData['data']['tmdb'] ?? [];
$type = strtolower($tmdb['type'] ?? 'movie');
$tmdb_id = $tmdb['tmdb_id'] ?? null;
$season = $season ?? ($tmdb['season'] ?? 1);
$episodeParam = $episode ?? 1;

// Prepare secondary API calls
$query = http_build_query([
    'tmdb_id' => $tmdb_id,
    'season'  => $season,
    'type'    => $type,
    'episode' => $episodeParam
]);

$allUrl  = BASE_API_URL . '/toonstream/all?'  . $query;
$saltUrl = BASE_API_URL . '/toonstream/salt?' . $query;

// Fetch data safely
$allData  = json_decode(@file_get_contents($allUrl), true);
$saltData = json_decode(@file_get_contents($saltUrl), true);

// ========================
// ✅ Extract sources properly for new format
// ========================
function extractSources($data, $season, $episode) {
    if (isset($data['episodes']) && is_array($data['episodes'])) {
        foreach ($data['episodes'] as $ep) {
            if ((int)$ep['season'] === (int)$season && (int)$ep['episode'] === (int)$episode) {
                return $ep['sources'] ?? [];
            }
        }
    }
    return $data['sources'] ?? [];
}

function extractDownloads($data, $season, $episode) {
    if (isset($data['episodes']) && is_array($data['episodes'])) {
        foreach ($data['episodes'] as $ep) {
            if ((int)$ep['season'] === (int)$season && (int)$ep['episode'] === (int)$episode) {
                return $ep['downloads'] ?? [];
            }
        }
    }
    return $data['downloads'] ?? [];
}

// ✅ Merge sources and downloads from both APIs
$allSources  = extractSources($allData, $season, $episodeParam);
$saltSources = extractSources($saltData, $season, $episodeParam);

$allDownloads  = extractDownloads($allData, $season, $episodeParam);
$saltDownloads = extractDownloads($saltData, $season, $episodeParam);

// ========================
// ✅ Deduplicate & merge sources
// ========================
$mergedSourcesRaw = [];
$seen = [];

foreach (array_merge($allSources, $saltSources) as $src) {
    if (is_array($src) && !empty($src['url'])) {
        $url = $src['url'];
        if (!isset($seen[$url])) {
            $mergedSourcesRaw[] = $url;
            $seen[$url] = true;
        }
    } elseif (is_string($src) && !isset($seen[$src])) {
        $mergedSourcesRaw[] = $src;
        $seen[$src] = true;
    }
}

$sources = $mergedSourcesRaw;

// ✅ Pull gdmirr list from detailsData
$episodes = $detailsData['data']['episodes'] ?? [];
$episodeGdmirrs = [];

if ($type === 'movie') {
    foreach ($episodes as $epData) {
        if (!empty($epData['gdmirr'])) {
            $episodeGdmirrs[] = [
                "resolution" => $epData['resolution'] ?? "Unknown",
                "url" => "https://piratexplay.cc/public/player/index.php?id={$epData['gdmirr']}",
                "label" => strtoupper($epData['resolution'] ?? "Unknown")
            ];
        }
    }
} else {
    foreach ($episodes as $epData) {
        if ((int)$epData['episode'] === (int)$episodeParam && !empty($epData['gdmirr'])) {
            $episodeGdmirrs[] = [
                "resolution" => $epData['resolution'] ?? "Unknown",
                "url" => "https://piratexplay.cc/public/player/index.php?id={$epData['gdmirr']}",
                "label" => strtoupper($epData['resolution'] ?? "Unknown")
            ];
        }
    }
}

// ✅ Helper: check if play link is valid
function isPlayableLink($url) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 8,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT => 'Mozilla/5.0'
    ]);
    $response = curl_exec($ch);
    curl_close($ch);

    if ($response === false) return false;
    return (stripos($response, 'Video not found') === false);
}

// ===== Categorize and label sources =====
$playSources = [];
$rubystmSources = [];
$abyssSources = [];
$otherSources = [];

foreach ($sources as $src) {
    $host = parse_url($src, PHP_URL_HOST);
    $label = null;

    if ($host) {
        $parts = explode('.', $host);
        $sub = strtolower($parts[0] ?? '');

    // Abyss (Animesalt)
    if (stripos($host, 'animesalt.cc') !== false) {
        // extract the ?data=... portion safely
        $parsedUrl = parse_url($src);
        $query = isset($parsedUrl['query']) ? $parsedUrl['query'] : '';
    
        // only keep the `data=` part
        parse_str($query, $params);
        $dataParam = isset($params['data']) ? $params['data'] : '';
    
        if (!empty($dataParam)) {
            // Rebuild your proxy link
            $proxyUrl = SITE_URL . "/proxy/multi.php?data=" . $dataParam;
    
            $label = 'ToonStream';
            $abyssSources[] = [
                "url" => $proxyUrl,
                "label" => $label
            ];
        }
    
        continue;
    }


        // Rubystm
        if (stripos($host, 'rubystm') !== false) {
            $label = 'Rubystm';
            $rubystmSources[] = ["url" => $src, "label" => $label];
            continue;
        }

        // Play-hosts first (but check validity)
        if (strpos($sub, 'play') !== false || stripos($host, 'play.') !== false) {
            if (isPlayableLink($src)) {
                $label = ucfirst($sub ?: $host);
                $playSources[] = ["url" => SITE_URL . "/proxy/play.php?url=" . $src, "label" => $label];
            }
            continue;
        }

        // Others
        $label = (count($parts) > 1) ? ucfirst($parts[0]) : ucfirst($host);
        $otherSources[] = ["url" => $src, "label" => $label];
    } else {
        // Handle malformed URLs
        if (preg_match('#^/?([^\.]+)\.#', $src, $m)) {
            $maybe = strtolower($m[1]);

            if (stripos($src, 'animesalt.cc') !== false || $maybe === 'animesalt') {
                $abyssSources[] = ["url" => $src, "label" => 'Abyss'];
                continue;
            }

            if (strpos($maybe, 'rubystm') !== false) {
                $rubystmSources[] = ["url" => $src, "label" => 'Rubystm'];
                continue;
            }

            if (strpos($maybe, 'play') !== false && isPlayableLink($src)) {
                $playSources[] = ["url" => $src, "label" => ucfirst($maybe)];
                continue;
            }
        }
        $otherSources[] = ["url" => $src, "label" => 'Unknown'];
    }
}

// Label gdmirr (piratex)
foreach ($episodeGdmirrs as &$g) {
    if (empty($g['label'])) $g['label'] = 'PIRATE';
}
unset($g);

// ✅ Final ordering: Play → Rubystm → Abyss → Piratex (gdmirr) → Others
$finalSources = array_merge($playSources, $rubystmSources, $abyssSources, $episodeGdmirrs, $otherSources);

// ✅ Output
$merged = [
    "status" => "success",
    "type" => $type,
    "title" => $tmdb['title'] ?? null,
    "image" => !empty($tmdb['poster']) ? "https://image.tmdb.org/t/p/w185{$tmdb['poster']}" : null,
    "season" => $season,
    "episode" => $episodeParam,
    "sources" => $finalSources,
    "downloads" => array_merge($allDownloads, $saltDownloads)
];

header('Content-Type: application/json');
echo json_encode($merged, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
