<?php
header('Content-Type: application/json');
require_once __DIR__ . '/../config.php';

// ✅ Validate input
if (empty($_GET['id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Missing id']);
    exit;
}

$id = trim($_GET['id']);
define('BASE_API_URL', rtrim(API_URL, '/'));

// 🔹 Fetch details for this specific season slug
$detailsUrl = BASE_API_URL . '/details/?id=' . urlencode($id);
$response = @file_get_contents($detailsUrl);

if (!$response) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to fetch API']);
    exit;
}

$json = json_decode($response, true);
if (!is_array($json) || empty($json['data'])) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid API response']);
    exit;
}

$data = $json['data'];
$tmdb = $data['tmdb'] ?? [];
$title = $tmdb['title'] ?? 'Unknown';
$tmdb_id = $tmdb['tmdb_id'] ?? $tmdb['id'] ?? null;
$seasonNum = (int)($tmdb['season'] ?? 1);

// 🔹 Fetch episode images
$episodeImages = [];
if (!empty($tmdb_id)) {
    $imgApiUrl = BASE_API_URL . "/toonstream/img?tmdb_id=" . urlencode($tmdb_id) . "&season={$seasonNum}";
    $imgResponse = @file_get_contents($imgApiUrl);
    $imgJson = json_decode($imgResponse, true);

    if (!empty($imgJson['images'])) {
        foreach ($imgJson['images'] as $ep) {
            $rawEp = trim($ep['episode'] ?? '');
            $imgUrl = trim($ep['image'] ?? '');
            if ($rawEp !== '' && $imgUrl !== '') {
                $num = (int)preg_replace('/\D+/', '', $rawEp);
                if ($num > 0) {
                    $episodeImages[$num] = htmlspecialchars($imgUrl, ENT_QUOTES, 'UTF-8');
                }
            }
        }
    }
}

// 🔹 Build episode list
$backdrop = !empty($tmdb['backdrop'])
    ? 'https://image.tmdb.org/t/p/w185' . $tmdb['backdrop']
    : (!empty($tmdb['poster'])
        ? 'https://image.tmdb.org/t/p/w185' . $tmdb['poster']
        : '/assets/no-image.jpg');

$episodesData = $data['episodes'] ?? [];
$episodes = [];
$seen = [];

// ✅ Collect unique, valid episodes
foreach ($episodesData as $ep) {
    $rawEp = trim($ep['episode'] ?? '');
    if ($rawEp === '') continue;

    $num = (int)preg_replace('/\D+/', '', $rawEp);
    if ($num <= 0) continue;

    // ✅ Skip duplicates safely
    if (isset($seen[$num])) continue;
    $seen[$num] = true;

    $episodes[] = [
        'season'  => $seasonNum,
        'episode' => $num,
        'image'   => $episodeImages[$num] ?? $backdrop,
    ];
}

// ✅ Sort numerically just once
// usort($episodes, fn($a, $b) => $a['episode'] <=> $b['episode']);

// ✅ Output clean JSON
echo json_encode([
    'status' => 'success',
    'id' => $id,
    'data' => [
        'tmdb' => $tmdb,
        'episodes' => $episodes
    ]
], JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
